import os
import csv
import json
import time
from datetime import datetime
import numpy as np

DATA_DIR = os.path.join(os.path.dirname(__file__), '..', 'data')
os.makedirs(DATA_DIR, exist_ok=True)

def load_csv_timeseries(path):
    """Load a simple two-column CSV (timestamp,value) where timestamp is ISO8601 or unix seconds."""
    times = []
    vals = []
    with open(path, 'r') as f:
        reader = csv.reader(f)
        headers = next(reader)
        for row in reader:
            if not row: continue
            t, v = row[0].strip(), row[1].strip()
            try:
                # try ISO8601
                ts = datetime.fromisoformat(t)
                ts_epoch = ts.timestamp()
            except Exception:
                try:
                    ts_epoch = float(t)
                except Exception:
                    ts_epoch = time.time()
            try:
                val = float(v)
            except Exception:
                val = np.nan
            times.append(ts_epoch)
            vals.append(val)
    return np.array(times), np.array(vals)

def save_metadata(outfile, metadata):
    with open(outfile, 'w') as f:
        json.dump(metadata, f, indent=2)

def ingest_csv(path, station_id='unknown', outname=None):
    times, vals = load_csv_timeseries(path)
    if outname is None:
        outname = f"{station_id}_{int(time.time())}.npz"
    outpath = os.path.join(DATA_DIR, outname)
    np.savez_compressed(outpath, times=times, values=vals)
    meta = {
        'station_id': station_id,
        'ingest_time': datetime.utcnow().isoformat() + 'Z',
        'samples': len(times),
        'format': 'npz'
    }
    meta_path = outpath + '.meta.json'
    save_metadata(meta_path, meta)
    return outpath, meta_path

if __name__ == '__main__':
    import argparse
    p = argparse.ArgumentParser()
    p.add_argument('csv', help='Path to CSV to ingest (timestamp,value)')
    p.add_argument('--station', default='unknown')
    args = p.parse_args()
    out, meta = ingest_csv(args.csv, station_id=args.station)
    print('Ingested:', out)
    print('Metadata:', meta)
